/**
* \file: message_sendr.c
*
* \version: $Id:$
*
* \release: $Name:$
*
* \component: automounter
*
* \author: Marko Hoyer / ADIT / SWGII / mhoyer@de.adit-jv.com
*
* \copyright (c) 2010, 2011 Advanced Driver Information Technology.
* This code is developed by Advanced Driver Information Technology.
* Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
* All rights reserved.
*
*
***********************************************************************/
#include "ipc/message_sendr.h"

// This value should fit for all messages sent during normal operation. The cache is automatically extended if
// a message is prepared for sending that is larger than the buffer size
#define DEFAULT_SEND_BUFFER_SIZE 512 //ensure selecting a value larger then sizeof(message_t)

//--------------------------------- private attributes -------------------------------------------------------------
static message_queue_t message_buffer_pool;

//--------------------------------- public member definition -------------------------------------------------------
error_code_t message_sendr_init(void)
{
	message_sendr_init_message_queue(&message_buffer_pool);
	return RESULT_OK;
}

void message_sendr_deinit(void)
{
	message_buffer_t *buf;
	while (message_buffer_pool.first_buffer != NULL)
	{
		buf=message_buffer_deque_from_begining(&message_buffer_pool);
		message_buffer_destroy(buf);
	}
}

message_buffer_t *message_sendr_get_empty_buffer(void)
{
	message_buffer_t *new_buffer=message_buffer_deque_from_begining(&message_buffer_pool);

	if (new_buffer==NULL)
		//queue is empty, create a new buffer
		new_buffer=message_buffer_create_new(DEFAULT_SEND_BUFFER_SIZE);
	return new_buffer;
}

void message_sendr_putback_buffer(message_buffer_t *buffer)
{
	message_buffer_mark_empty(buffer);
	message_buffer_insert_at_begining(&message_buffer_pool,buffer);
}

error_code_t message_sendr_prepare_message(message_buffer_t *buffer, message_type_t message_type,
		size_t data_size, void **data_buffer)
{
	return message_buffer_prepare_message(buffer,message_type,data_size,data_buffer);
}

error_code_t message_sendr_add_string_to_msg(message_buffer_t *buffer, const char *string)
{
	return message_buffer_add_string(buffer,string);
}

error_code_t message_sendr_send_buffer(message_buffer_t *buffer, int a_socket)
{
	return message_buffer_send(buffer, a_socket);
}

error_code_t message_sendr_clone_msg_in_buffer(message_buffer_t *dst_buffer, message_buffer_t *src_buffer)
{
	return message_buffer_clone_msg_in_buffer(dst_buffer, src_buffer);
}

void message_sendr_init_message_queue(message_queue_t *queue)
{
	message_buffer_init_queue(queue);
}

void message_sendr_add_buffer_to_send_queue(message_buffer_t *buffer, message_queue_t *queue)
{
	message_buffer_insert_at_begining(queue, buffer);
}

error_code_t message_sendr_send_queued_buffers(message_queue_t *queue, int a_socket)
{
	error_code_t result=RESULT_OK;
	message_buffer_t *buf;
	buf=queue->last_buffer;
	while(result==RESULT_OK && buf!=NULL)
	{
		result=message_sendr_send_buffer(buf,a_socket);
		if (result==RESULT_OK)
		{
			//we sent the buffer, dequeue it and put it back into the pool
			message_buffer_remove_from_queue(queue, buf);
			message_sendr_putback_buffer(buf);
			buf=queue->last_buffer;
		}
	}

	return result;
}

void message_sendr_empty_queue(message_queue_t *queue)
{
	message_buffer_t *buf;
	while (queue->first_buffer != NULL)
	{
		buf=message_buffer_deque_from_begining(queue);
		message_sendr_putback_buffer(buf);
	}
}
